﻿// Default Using Statements
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

// Added System Using Statements
using System.Threading;
using System.IO;
using System.Diagnostics;

// Hardware control Using Statements
using Ivi.Visa.Interop;
#if aemulus
using Aemulus.Hardware.SMU;
#endif

namespace PowerAmpDemoNS
{
    public class DcSmu
    {
        // Data log from creating object
        private DataLog dataLog;

        // DC SMU Variables
        public bool useDc;
        public bool useN6700 = false;
        public bool useAm4xx = false;
        ResourceManager oRm;
        FormattedIO488 n6700Io;
        public string dcResource = "COM5";
        // Only create the Aeumulus Objects if the conditional compliation symbol is defined
#if aemulus
        public PxiSmu AM471_1   ;
        public PxiSmu AM471_2;
#endif
        // Arrays for DC SMU Setup information
        public double[] dcVoltage;
        public double[] dcCurrLim;
        public bool[] enableDcChan;
        public bool[] doCurrMeas;
        public string n6700MeasList;
        private Stopwatch sw1 = new Stopwatch();

        public DcSmu(DataLog parentDataLog)
        {
            dataLog = parentDataLog;
        }

        public void initDcSmus()
        {
            if (!useDc) return;
            // DC Module Initialization Here
            if (useAm4xx)
            {
#if aemulus
                    AM471_1 = new PxiSmu("PXI20::0::0::INSTR", "0", 0xf, "Simulate=0, DriverSetup=Model:am471e");
                    AM471_2 = new PxiSmu("PXI21::0::0::INSTR", "0", 0xf, "Simulate=0, DriverSetup=Model:am471e");
#endif
            }
            else if (useN6700)
            {
                if (oRm == null)
                    oRm = new ResourceManager();
                n6700Io = new FormattedIO488();
                //Open session for instrument.
                n6700Io.IO = (IMessage)oRm.Open(dcResource, AccessMode.NO_LOCK, 2000, "");
                n6700Io.IO.Timeout = 10000;
                n6700Io.WriteString("*RST");
                n6700Io.WriteString("SYSTEM:ERROR?\n");
                string s;
                s = n6700Io.ReadString();
                if (!s.ToUpper().Contains("NO ERROR")) throw (new Exception("N6700 Error:\n" + s));

            }
        }

        // readDcSetupData
        // Reads a DC SMU data file with the following format:
        /*
                Channels,4,
                ,,
                Setup Data,,
                Enable Channel,Voltage,Current Limit, Measure Current
                1,3.0, 1.0, 1
                1,3.0, 1.0, 1
                0,3.0, 1.0, 0
                0,3.0, 1.0, 0
        */
        public void readDcSetupData(string filePath, string fileName)
        {
            int numChannels = 0;

            // Clean up any old DC Setup data
            if (dcVoltage != null) dcVoltage = null;
            if (dcCurrLim != null) dcCurrLim = null;
            if (enableDcChan != null) enableDcChan = null;
            if (doCurrMeas != null) doCurrMeas = null;

            // Open the file
            var rs = new FileStream(filePath + fileName, FileMode.Open);
            TextReader rd = new StreamReader(rs);

            string fileString;
            // Read the data size values
            while (numChannels == 0)
            {
                fileString = rd.ReadLine();
                fileString = fileString.ToUpper();
                string[] columns = fileString.Split(',');
                if (columns[0].Contains("CHANNELS")) numChannels = Convert.ToInt32(columns[1]);
            }

            // Size the DC Setup arrays
            dcVoltage = new double[numChannels];
            dcCurrLim = new double[numChannels];
            enableDcChan = new bool[numChannels];
            doCurrMeas = new bool[numChannels];

            bool setupDataRead = false;
            while (!setupDataRead)
            {
                fileString = rd.ReadLine();
                fileString = fileString.ToUpper();
                if (fileString.Contains("SETUP DATA"))
                {
                    int linesRead = 0;
                    while (linesRead < numChannels)
                    {
                        fileString = rd.ReadLine();
                        fileString = fileString.ToUpper();
                        if (fileString.Contains("ENABLE") || fileString.Contains("//"))
                        {
                            // Skip Header or comment line
                        }
                        else
                        {
                            // Read the setup data
                            string[] columns = fileString.Split(',');
                            enableDcChan[linesRead] = Convert.ToBoolean(columns[0]);
                            dcVoltage[linesRead] = Convert.ToDouble(columns[1]);
                            dcCurrLim[linesRead] = Convert.ToDouble(columns[2]);
                            doCurrMeas[linesRead] = Convert.ToBoolean(columns[3]);
                            linesRead++;
                        }
                    }
                    setupDataRead = true;
                }
            }
            rd.Close();
            rs.Close();
        }

        public void setupSmus()
        {
            if (!useDc) return;
            // DC Module Setup Here
            if (useAm4xx)
            {
#if aemulus
                AM471_1.ConfigureOutputFunction("0", PxiSmuConstants.DVCI);
                AM471_1.ConfigureCurrentLimitAndRange("0", 0, .5, 1);
                AM471_1.ConfigureVoltageLevel("0", 8.0);
                AM471_1.ConfigureOutputTransient("0", 2);
                AM471_1.ConfigureSense("0", PxiSmuConstants.SenseLocal);

                AM471_2.ConfigureOutputFunction("0", PxiSmuConstants.DVCI);
                AM471_2.ConfigureCurrentLimitAndRange("0", 0, .5, 1);
                AM471_2.ConfigureVoltageLevel("0", 8.0);
                AM471_2.ConfigureOutputTransient("0", 2);
                AM471_2.ConfigureSense("0", PxiSmuConstants.SenseLocal);
#endif
            }
            else if (useN6700)
            {
                n6700MeasList = "";
                string cmd;
                int chans = dcVoltage.Length;
                for (int i = 0; i < chans; ++i)
                {
                    if (enableDcChan[i])
                    {
                        cmd = string.Format("VOLT {0},(@{1})", dcVoltage[i], i + 1); // set voltage
                        n6700Io.WriteString(cmd);
                        cmd = string.Format("CURR:LIM {0},(@{1})", dcCurrLim[i], i + 1); // set current limit
                        n6700Io.WriteString(cmd);
                        cmd = string.Format("CURR:PROT:STAT OFF,(@{0})", i + 1); // enable current limit
                        n6700Io.WriteString(cmd);
                        if (doCurrMeas[i])
                        {
                            // Setup a nominal 1000 us current measurement time
                            cmd = string.Format("SENS:SWE:POIN {1},(@{0})", i + 1, "MIN"); // enable current limit
                            n6700Io.WriteString(cmd);
                            cmd = string.Format("SENS:SWE:TINT {1},(@{0})", i + 1, .001); // enable current limit
                            n6700Io.WriteString(cmd);
                            cmd = string.Format("SENS:FUNC:CURR ON,(@{0})", i + 1); // enable current limit
                            n6700Io.WriteString(cmd);
                            // Build a string of the channels to measure
                            n6700MeasList = string.Format("{0}{1},", n6700MeasList, i + 1);
                        }
                    }
                }
                // Remove the last "," from the channel list
                n6700MeasList = n6700MeasList.Substring(0, n6700MeasList.Length - 1);
                n6700Io.WriteString("SYSTEM:ERROR?\n");
                string s;
                s = n6700Io.ReadString();
                if (!s.ToUpper().Contains("NO ERROR")) throw (new Exception("N6700 Error:\n" + s));
            }
        }

        public void openSmuOutputRelays()
        {
            // Open the Output Relays
            if (!useDc) return;
            // DC Module Setup Here
            if (useAm4xx)
            {
#if aemulus
                    AM471_1.ConfigureOutputSwitch("0", 0);
                    AM471_2.ConfigureOutputSwitch("0", 0);
#endif
            }
            else if (useN6700)
            {
                string cmd;
                int chans = dcVoltage.Length;
                for (int i = 0; i < chans; ++i)
                {
                    if (enableDcChan[i])
                    {
                        cmd = string.Format("OUTP OFF,(@{0})", i + 1); // turn off SMU
                        n6700Io.WriteString(cmd);
                    }
                }
                n6700Io.WriteString("SYSTEM:ERROR?\n");
                string s;
                s = n6700Io.ReadString();
                if (!s.ToUpper().Contains("NO ERROR")) throw (new Exception("N6700 Error:\n" + s));
            }
        }

        public void closeSmuOutputRelays()
        {
            // Close the Output Relays
            if (!useDc) return;
            // DC Module Setup Here
            if (useAm4xx)
            {
#if aemulus
                    AM471_1.ConfigureOutputSwitch("0", 1);
                    AM471_2.ConfigureOutputSwitch("0", 1);
#endif
            }
            else if (useN6700)
            {
                string cmd;
                int chans = dcVoltage.Length;
                for (int i = 0; i < chans; ++i)
                {
                    if (enableDcChan[i])
                    {
                        cmd = string.Format("OUTP ON,(@{0})", i + 1); // turn on SMU
                        n6700Io.WriteString(cmd);
                    }
                }
                n6700Io.WriteString("SYSTEM:ERROR?\n");
                string s;
                s = n6700Io.ReadString();
                if (!s.ToUpper().Contains("NO ERROR")) throw (new Exception("N6700 Error:\n" + s));
            }
        }

        public void closeDcSmus()
        {
            if (!useDc) return;
            // DC Modules Close Driver here'
            if (useAm4xx)
            {

#if aemulus
                    if (AM471_1 != null)
                    {
                        AM471_1.ConfigureOutputSwitch("0", 0);
                        AM471_1.Dispose();
                        AM471_1 = null;
                    }
                    if (AM471_2 != null)
                    {
                        AM471_2.ConfigureOutputSwitch("0", 0);
                        AM471_2.Dispose();
                        AM471_2 = null;
                    }
#endif
            }
            else if (useN6700)
            {
                openSmuOutputRelays();
                n6700Io.WriteString("SYSTEM:ERROR?\n");
                string s;
                s = n6700Io.ReadString();
                if (!s.ToUpper().Contains("NO ERROR")) throw (new Exception("N6700 Error:\n" + s));
            }
        }

        public void measCurrent()
        {
            int numMeas = 0;

            //Start Test timer
            sw1.Reset();
            sw1.Start();
            // Measure DC curretn
            if (!useDc) return;

            if (useAm4xx)
            {
#if aemulus
                    AM471_1.ConfigureOutputSwitch("0", 1);
                    AM471_2.ConfigureOutputSwitch("0", 1);
#endif
            }
            else if (useN6700)
            {
                string cmd;
                string s;
                int chans = dcVoltage.Length;
                if (n6700MeasList.Length == 0) return;
                // Measure all of the channels together
                cmd = string.Format("MEAS:CURR? (@{0})", n6700MeasList); // Measure
                n6700Io.WriteString(cmd);
                s = n6700Io.ReadString();
                string[] columns = s.Split(',');
                for (int i = 0; i < chans; ++i)
                {
                    if (doCurrMeas[i])
                    {
                        dataLog.measCurr[dataLog.testIndex, i] = Convert.ToDouble(columns[numMeas]);
                        numMeas++;
                    }
                }
                n6700Io.WriteString("SYSTEM:ERROR?\n");
                s = n6700Io.ReadString();
                if (!s.ToUpper().Contains("NO ERROR")) throw (new Exception("N6700 Error:\n" + s));
                long localMeasTime = sw1.ElapsedMilliseconds;

                // Log Results
                string[] dataNames = new string[numMeas + 1];
                string[] dataUnits = new string[numMeas + 1];
                string[] numFormats = new string[numMeas + 1];
                double[] dataValues = new double[numMeas + 1];

                int j = 0;
                for (int i = 0; i < chans; i++)
                {
                    if (doCurrMeas[i])
                    {
                        dataNames[j] = "DC Chan " + (i + 1).ToString();
                        dataUnits[j] = "mAmps";
                        numFormats[j] = "0.0000";
                        dataValues[j] = dataLog.measCurr[dataLog.testIndex, i] * 1000;
                        j++;
                    }
                }
                dataNames[dataNames.Length - 1] = "Curr Time";
                dataUnits[dataNames.Length - 1] = "ms";
                numFormats[dataNames.Length - 1] = "0";
                dataValues[dataNames.Length - 1] = sw1.ElapsedMilliseconds;
                dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
            }
        }

    }
}
